/* @file Logger.cpp
*
*  @brief A singleton class responsible for creating single instance of Logger
*
*  @author CppBuzz.com
*  @Date 10th Aug 2022
*  @version 1.0
*
*  Note : This class is not thread safe.
*/

#ifndef _LOGGER_H
#define _LOGGER_H

#include "FileLogger.h"

#include <cstdio>
#include <string>
#include <cstdarg>
#include <cstring>
#include <algorithm>
#include <sstream>

using namespace std;


class Logger
{

public:

    /* enum to set loglevel */
    typedef enum
    {
        NONE=0,         /* LogLevel is set to NONE if not logging required*/
        ERROR,
        WARNING,
        DEBUG,
        INFO

    }LogLevel;

    /*  Stream to save current Stream
        Logs can be send to File, stdout, stderr streams
    */
    typedef enum
    {
        NO_STREAM=0,       /* Stream can be set to NO_STR is no logging required */
        FILE_STREAM,
        STDOUT_STRREAM,
        STDERR_STREAM

    }Stream;

    /*  @brief Function to get instance of Logger
        @return Instance of Logger
        @params Stream 
        @params LogLevel
        @params string
    */
    static Logger* getLogHandler(Stream stream = Logger::FILE_STREAM,
                                     LogLevel logLevel = Logger::INFO,
                                     const string& fileName = "");
   
    /*  @brief Function to write logs to stream selected
        @return void
        @params Loglevel, string
    */
    void writeLog(Logger::LogLevel logLevel, const char * format = "", ...);

    /* Function to set loglevel */
    static void setLogLevel(LogLevel logLevel);

    /*  @brief Function to set loglevel using string
        @return bool
        @params string
    */
    static bool setLogLevel(string logLevel);

    /*  @brief Function to get current loglevel
        @return LogLevel
    */
    static LogLevel getLogLevel();
   
    /* Destructor not doing any operation */
    ~Logger();

private:

    /* m_eLoglevel is current loglevel */
    static LogLevel m_eLogLevel;

    /* m_eStream is current Stream(File, stdOut, stdErr) */
    static Stream m_eStream;

    /* Constructor to set stream */
    Logger(Stream stream, const string& fileName);

    /* Copy Constructor - Not required to be implemented */
    Logger(Logger const& copy);

    /* Assignment Opertor - Not requird to be implemented */
    Logger& operator=(Logger const& assign);

    /*  @brief Local function to convert LogLevel into string
        @return string
        @params Loglevel
    */
    string getStringLogLevel(LogLevel& loglevel);

    /* Pointer which pointes to specified stream */
    ILogWriter* m_pLogWriter;

    /* Constants for log level */
    static const string LOGGER_LOG_LEVEL_STRING_NONE;
    static const string LOGGER_LOG_LEVEL_STRING_ERROR;
    static const string LOGGER_LOG_LEVEL_STRING_WARNING;
    static const string LOGGER_LOG_LEVEL_STRING_DEBUG;
    static const string LOGGER_LOG_LEVEL_STRING_INFO;

};

#endif /* End of _LOGGER_H */
